//////////////////////////////////////////////////////
//
//	ClusterManager.java
//
//
package Alkindi.Services.ServicesImpl;

import java.util.*;
import java.io.*;
import java.sql.*;
import javax.ejb.*;
import javax.transaction.*;
import Alkindi.Data.*;
import Alkindi.Services.*;
import Alkindi.Services.Util.*;
import Alkindi.Services.InternalData.*;

/* 
$Header: ClusterManager.java, 78, 4/26/01 2:37:15 PM, Schwartz, Joe$
$Log: 
 78   Alkindi Development1.77        4/26/01 2:37:15 PM   Schwartz, Joe  
      Modifed to account for new InternalData package.
 77   Alkindi Development1.76        4/26/01 1:11:12 PM   Schwartz, Joe   Moved
      into new ServicesImpl package.
 76   Alkindi Development1.75        4/25/01 4:41:59 PM   Schwartz, Joe  
      Removed use of transactions due to problems with transaction timeout.
 75   Alkindi Development1.74        4/23/01 10:51:54 AM  Schwartz, Joe   Added
      start and stop log notifications to recalcStats and regenClusters.
 74   Alkindi Development1.73        4/18/01 1:50:56 PM   Schwartz, Joe   Fixed
      error reporting in regenerateClusters().
 73   Alkindi Development1.72        3/16/01 12:31:12 PM  Schwartz, Joe   Worked
      to fix error in which bad recalced vmeans led to curtailed filling.
 72   Alkindi Development1.71        3/14/01 5:37:31 PM   Schwartz, Joe  
      Changed to abstract class. Added abstract method getXA to retrieve
      transaction object. Removed sessCtx object.
 71   Alkindi Development1.70        2/27/01 6:09:27 PM   Schwartz, Joe  
      Changed to save only UCs with members during initClusters.
 70   Alkindi Development1.69        2/27/01 1:01:40 PM   Schwartz, Joe  
      Maintainance checkin.
 69   Alkindi Development1.68        2/23/01 4:33:56 PM   Schwartz, Joe   Added
      writing of filled in product UC stats.
 68   Alkindi Development1.67        2/22/01 5:24:07 PM   Schwartz, Joe   Added
      explicit transaction to regenerateClusters method.
 67   Alkindi Development1.66        2/22/01 3:16:17 PM   Schwartz, Joe  
      Changed to support common matrix routines in VMeansList. Fixed some errors
      in filling & regenClusters.
 66   Alkindi Development1.65        2/15/01 5:47:03 PM   Schwartz, Joe   Added
      debugProdFill property to limit recalcStats to fillInProdUCStats.
 65   Alkindi Development1.64        2/14/01 6:22:31 PM   Schwartz, Joe  
      Changed to accomodate new util classes.
 64   Alkindi Development1.63        2/13/01 5:55:29 PM   Schwartz, Joe  
      Changed to account for new Product id int type and SparseRatingsArray.
 63   Alkindi Development1.62        2/12/01 10:38:09 AM  Schwartz, Joe   Added
      instrumentation to regenClusters.
 62   Alkindi Development1.61        2/6/01 11:53:47 AM   Schwartz, Joe   Adding
      new Product filling.
 61   Alkindi Development1.60        2/1/01 4:03:39 PM    Schwartz, Joe  
      Regenerated code from Rose with correct imports.
 60   Alkindi Development1.59        1/31/01 11:37:39 AM  Schwartz, Joe  
      Generated with new Rose model.
 59   Alkindi Development1.58        1/31/01 11:21:38 AM  Schwartz, Joe  
      Generated by new Rose model.
 58   Alkindi Development1.57        1/26/01 7:45:44 PM   Schwartz, Joe   Spped
      optimizations.
 57   Alkindi Development1.56        1/26/01 5:57:41 PM   Schwartz, Joe  
      Simplified clusterUsers.
 56   Alkindi Development1.55        1/26/01 5:48:39 PM   Schwartz, Joe  
      Working on speed.
 55   Alkindi Development1.54        1/25/01 9:58:05 PM   Schwartz, Joe  
      Working on speed, sensibility.
 54   Alkindi Development1.53        1/22/01 6:47:57 PM   Schwartz, Joe   TEMP:
      Working on clustering speed.
 53   Alkindi Development1.52        1/22/01 1:46:46 PM   Schwartz, Joe  
      Modified to account for new RatingSpacePoint base class of VMeans &
      UserCluster.
 52   Alkindi Development1.51        1/21/01 4:43:04 PM   Schwartz, Joe   Made
      several changes to fill function to improve speed. Changed
      fillPointsFromNeighbors to fillPointFromNeighbors: now fills only one
      point, so we avoid extra iteration over points; changed distance matrix
      calculation to exploit symmetry.
      Also, added environment properties to control instrumentation and what PCs
      are considered in making User Clusters.
 51   Alkindi Development1.50        1/20/01 6:37:54 PM   Schwartz, Joe  
      Optimized calc of distance matrix in fillFunction. Added use of startPC
      and stopPC properties.
 50   Alkindi Development1.49        1/4/01 4:40:06 PM    Schwartz, Joe  
      Removed references to transaction pamameter in calls to recalcXStats.
 49   Alkindi Development1.48        1/4/01 2:12:27 PM    Schwartz, Joe  
      Changed to pass LogManager instance to CDB constructor. Changed
      transaction handling in recalcStats.
 48   Alkindi Development1.47        1/2/01 4:30:18 PM    Schwartz, Joe  
      Removed assignUserToNearestCluster; replaced with simple data
      manipulation.
 47   Alkindi Development1.46        1/2/01 4:28:14 PM    Schwartz, Joe   Added
      call to fillFunction for VMeans list before storing it in the database.
 46   Alkindi Development1.45        12/28/00 2:46:32 PM  Schwartz, Joe  
      Standardized methods to use final parameters and fully-qualified types
      where necessary. Makes Rose happy. Also, made most methods protected.
 45   Alkindi Development1.44        12/28/00 1:47:10 PM  Schwartz, Joe  
      Replaced missing methods and other errors due to Rose import. 
 44   Alkindi Development1.43        12/28/00 1:09:22 PM  Schwartz, Joe   Added
      Version Control header info.
      Synchronized with IClusterManager interface.
      Changed package-scope members to protected.
 43   Alkindi Development1.42        12/27/00 6:10:11 PM  Schwartz, Joe   
 42   Alkindi Development1.41        12/26/00 6:57:10 PM  Schwartz, Joe   
 41   Alkindi Development1.40        12/26/00 5:36:58 PM  Schwartz, Joe   
 40   Alkindi Development1.39        12/22/00 5:30:23 PM  Schwartz, Joe   
 39   Alkindi Development1.38        12/21/00 2:24:26 PM  Schwartz, Joe   Fixed
      fillFunction convergence. Changed cluster convergence test to use 
      unfilled VMeans. Added better XAction mgt.
 38   Alkindi Development1.37        12/20/00 5:44:53 PM  Schwartz, Joe   
 37   Alkindi Development1.36        12/18/00 4:35:38 PM  Schwartz, Joe   
 36   Alkindi Development1.35        12/18/00 12:05:34 PM Schwartz, Joe   Moved
      internal data classes to Utils package & regenerated classes from Rose.
 35   Alkindi Development1.34        12/18/00 11:22:08 AM Schwartz, Joe   
 34   Alkindi Development1.33        12/7/00 5:02:29 PM   Schwartz, Joe   
 33   Alkindi Development1.32        12/6/00 7:11:57 PM   Schwartz, Joe   Added
      catch blocks for generic exceptions and finally blocks to close all
      database connections.
 32   Alkindi Development1.31        12/6/00 11:27:50 AM  Schwartz, Joe   Added
      modified fill function per ES.
 31   Alkindi Development1.30        12/3/00 11:40:18 PM  Schwartz, Joe   Now
      initClusters skips fillFunction, since it tends to force all users into
      the first user cluster.
 30   Alkindi Development1.29        12/3/00 7:51:25 PM   Schwartz, Joe   
 29   Alkindi Development1.28        12/3/00 5:54:00 PM   Schwartz, Joe  
      Modified to use new QuickProdStats/QuickProdStatsList/UserRatings objects
      that are now based on TreeMap instead of Vector for faster searches.
 28   Alkindi Development1.27        12/3/00 5:27:55 PM   Schwartz, Joe   
 27   Alkindi Development1.26        12/3/00 3:55:37 PM   Schwartz, Joe   
 26   Alkindi Development1.25        12/2/00 9:21:49 PM   Schwartz, Joe   
 25   Alkindi Development1.24        12/2/00 8:31:42 PM   Schwartz, Joe   
 24   Alkindi Development1.23        12/2/00 7:37:25 PM   Schwartz, Joe   
 23   Alkindi Development1.22        12/2/00 2:57:40 PM   Schwartz, Joe   
 22   Alkindi Development1.21        12/2/00 1:02:26 PM   Schwartz, Joe   
 21   Alkindi Development1.20        12/2/00 10:47:12 AM  Schwartz, Joe   
 20   Alkindi Development1.19        11/30/00 12:46:42 AM Schwartz, Joe   
 19   Alkindi Development1.18        11/28/00 5:51:13 PM  Schwartz, Joe  
      Attempting to fix bugs in clustering.
 18   Alkindi Development1.17        11/27/00 7:40:11 PM  Schwartz, Joe   
 17   Alkindi Development1.16        11/27/00 7:29:09 PM  Schwartz, Joe  
      Attempted to fix major clustering errors.
 16   Alkindi Development1.15        11/20/00 10:52:34 PM Schwartz, Joe   
 15   Alkindi Development1.14        11/14/00 12:53:12 PM Schwartz, Joe   
 14   Alkindi Development1.13        11/3/00 1:29:57 PM   Schwartz, Joe   
 13   Alkindi Development1.12        11/2/00 12:33:21 PM  Schwartz, Joe   
 12   Alkindi Development1.11        10/31/00 7:04:45 PM  Schwartz, Joe   
 11   Alkindi Development1.10        10/30/00 7:52:22 PM  Schwartz, Joe   
 10   Alkindi Development1.9         10/22/00 2:38:10 PM  Schwartz, Joe   Added
      instrumentation.
 9    Alkindi Development1.8         10/22/00 10:38:40 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 8    Alkindi Development1.7         10/22/00 10:34:08 AM Schwartz, Joe   
 7    Alkindi Development1.6         10/20/00 1:56:54 PM  Schwartz, Joe   
 6    Alkindi Development1.5         10/20/00 1:11:30 PM  Schwartz, Joe   
 5    Alkindi Development1.4         10/20/00 12:01:25 PM Schwartz, Joe  
      Delivery 10.19.00 PM
 4    Alkindi Development1.3         10/20/00 12:00:44 PM Schwartz, Joe   
 3    Alkindi Development1.2         10/17/00 2:22:56 PM  Schwartz, Joe  
      Delivery 10.16.00
 2    Alkindi Development1.1         10/7/00 4:34:29 PM   Schwartz, Joe   
 1    Alkindi Development1.0         10/7/00 4:26:02 PM   Schwartz, Joe   
$
$NoKeywords$
 */

/**
 */
public abstract class ClusterManager implements IClusterManager 
{
	protected static int NUM_CLUSTERS = 15;
	
	/**
	 * 	Tolerance used in fill function (an Engine property).
	 */
	protected static double propDFTr;
	
	/**
	 * Controls the measurement and display of instrumentation info.
	 */
	protected boolean propInstrumentation;
	
	/**
	 * 	Maximum iterations to use in fill function (an Engine property).
	 */
	protected static int propMaxDFTIter;
	protected static double propMaxRating;
	
	/**
	 * 	Number of Inner Loops to use in clustering (an Engine property).
	 */
	protected static int propNumInnerLoops;
	
	/**
	 * 	Number of Product Clusters (an Engine property).
	 */
	protected static int propNumPCs;
	
	/**
	 * 	Number of Uber Loops to use in clustering (an Engine property).
	 */
	protected static int propNumUberLoops;
	
	/**
	 * 	Number of users who may have rated a product for it to be considered in some stat calculations (an Engine property).
	 */
	protected static int propOmega;
	
	/**
	 * 	Power to use in distance calcuations (an Engine property).
	 */
	protected static float propPower;
	
	/**
	 * Product Cluster ID at which to begin clustering.
	 */
	protected int propStartPC;
	
	/**
	 * Product Cluster ID at which to stop clustering.
	 */
	protected int propStopPC;
	
	/**
	 * 	Number of top-scoring products to consider in some stat calculations (an Engine property).
	 */
	protected static int propTopNProds;
	
	/**
	 *  Tolerance used in determining cluster VMeans convergence (an Engine property).
	 */
	protected static double propTr;
	protected static boolean propUseRandomVMeans;
	static final String cName = "ClusterManager";
	
	/**
	 *  CM Database utilities
	 */
	ClusterMgrDBUtils cdb = null;
	
	/**
	 * Logging object
	 */
	LogManager logger = null;
	
	/**
	 * Assigns the given user to a cluster. This is the entry point to "fast" clustering.
	 * @param user The SystemUser to cluster.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B6609029F
	 */
	public void clusterUser(final Alkindi.Data.SystemUser user) throws AlkExcept 
	{  
		cdb.clusterSingleUser(user);
	}
	
	/**
	 * Assigns all the users in thegiven UserRatingsList into UserCluster clusters whose centers are the given VMeansList.
	 * This is the lengthiest operation in the clustering process, as it must iterate over all users & their rating for each UserCluster.
	 * @param vml The VMeans list of UserCluster centers.
	 * @param url The UserRatingsList containing all the 
	 * @return UserClusterList
	 * @throws AlkExcept.
	 * @roseuid 3A4B660902BF
	 */
	protected final UserClusterList clusterUsers(VMeansList vml, UserRatingsList url) throws AlkExcept 
	{
		final String mName = "clusterUsers";
		Instrumenter instr = null;
		if (propInstrumentation) {
			instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
		}

		VMeans      	pMean;

		// Initialize a Vector for the List of VMeans  
		//
		pMean		= (VMeans) vml.get(0);    

		// Init. an array for distance in 1-dimension 
		//
		double[] distmetric = new double[vml.size()];    

		// Init. an array for sum of exponentiated distances for all dimensions
		//
		double []   		sumpowermetric 	= new double[vml.size()];    

		// 	Declare UserClusterList to return
		//
		UserClusterList ucl = UserClusterList.getInstance(vml.size());
		
		int numVMeans = vml.size();
		
		//	Initialize the User Clusters by iterating over the vmeans.
		//
		for (int clusIdx = 0; clusIdx < numVMeans; clusIdx++) {
			ucl.addCluster(vml.get(clusIdx), url.products);
		}
		// 	Iterate through all users by iterating through all UserRatings 
		//	objects stored in the UserRatingsList.
		//
		Iterator userRatingsIt = url.userRatingsIterator();
		Iterator prodIt = null;
		while (userRatingsIt.hasNext()) {
			UserRatings ur = (UserRatings)userRatingsIt.next();    
			int chosenCluster = 0;
			float minDist = 0f;
			for (int vmIdx = 0; vmIdx < numVMeans; vmIdx++) {
				VMeans vm = vml.get(vmIdx);
				float dist = ur.distance(vm, propPower);
				if (vmIdx == 0 || dist < minDist) {
					minDist = dist;
					chosenCluster = vmIdx;
				}
			}
			ucl.addUserToCluster(ur, chosenCluster);
		}//	END Iterate over users / UserRatings

		//logger.dbgLog(cName, mName, ucl.dump());
		if (propInstrumentation)
			instr.stopTiming();
		return ucl;
	}
	
	/**
	 * The data filling function. This constructs data for VMeans points which are missing evaluations for particular products ad returns a VMeansList of filled points. 
	 * To create data for a particular product of a given UserCluster, it uses the "neighboring" points' evaluations for that product. The neighboring points are taken to be the VMeans for other UserClusters.It weights these evaluations both by the number of members of those other UserClusters and by their proximity to the UserCluster being filled.
	 * @param points The VMeans to fill.
	 * @param ucl The UserClusterList used to determine weights of the neighbors.
	 * @return VMeansList
	 * @throws AlkExcept
	 * @roseuid 3A4B660A003E
	 */
	protected final VMeansList fillFunction(VMeansList points, UserClusterList ucl) throws AlkExcept 
	{
		final String mName = "fillFunction";
		Instrumenter instr = null;
		if (propInstrumentation) {
			instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
		}
		
		int numProds =  points.get(0).numProducts();
		int numPoints = points.size();
		VMeansList newPoints = null;
		int numIter = 0;
		boolean iterateAgain = true;
		double[] clusMemVector = new double[numPoints];

		
		//	If a null reference to a UserClusterList was passed, fill clusMemVector with 1.
		//	Otherwise fill clusMemVector with the number of users in each cluster.
		//
		if (ucl == null) {
			Arrays.fill(clusMemVector, 1.0);
		}
		else {
			int numClusters = ucl.size();
			for (int ucIdx = 0; ucIdx < numClusters; ucIdx ++) {
				clusMemVector[ucIdx] = ucl.get(ucIdx).numUsers();
			//logger.dbgLog(cName, mName, "clusMemVector[" + (ucIdx-1) + "]=" + clusMemVector[ucIdx - 1]);
			}
		}

		// 	Record which points are missing data for which products.
		//
		boolean[][] sparsityMatrix = points.getSparsityMatrix();
		//logger.dbgLog(cName, mName, "Finding points with missing data.");

		// 	Declare weights 
		//
		double[] weights = new double[numPoints];

		// Initialize distanceMatrix.
		//
		double[][] distanceMatrix = new double[numPoints][numPoints];
		for (int ipt=0; ipt < numPoints; ipt++) {
			Arrays.fill(distanceMatrix[ipt], 1);
		}

		// Begin main loop
		//
		//logger.dbgLog(cName, mName, "Main loop starting");
		while (iterateAgain && numIter < propMaxDFTIter) {
			newPoints = (VMeansList)points.clone();
	
			iterateAgain = false;
			numIter++;

			// Construct weightMatrix to pass to fillPoint
			//
			// Loop over points in which to fill data
			//
			for (int ipt=0; ipt<numPoints; ipt++) {
				//logger.dbgLog(cName, mName, "point loop " + numIter);
				//
				for (int ipt2=0; ipt2 < numPoints; ipt2++) {
					//logger.dbgLog(cName, mName, "Entering inter-point loop: " + ipt2);
					if (ipt2 == ipt) {
						weights[ipt2] = 0;
					} else {
						// Weight of ipt2 proportional to cluster membership, inversely
						// proportion to distance between ipt2 and ipt.
						//	But if distance matrix element is zero, weight should also be zero.
						//
						if (distanceMatrix[ipt][ipt2] == 0) {
							weights[ipt2]	= 0;
						}
						else {
							weights[ipt2] = (double)clusMemVector[ipt2] / distanceMatrix[ipt][ipt2];
						}
					} // End if (ipt2 == ipt)
					//logger.dbgLog(cName, mName, "Weight[" + ipt + ", " + ipt2 + "]=" + weights[ipt2]);
				} // End loop over points from which we fill data (ipt)
				boolean brokeTol = fillPoint(newPoints.get(ipt), points, sparsityMatrix, weights, ipt);
				if (brokeTol) iterateAgain = true;


			} // End loop over points in which we fill data (ipt)

/*			for (int prodIdx = 0; prodIdx < numProds; prodIdx++) {
				for(int ptIdx = 0; ptIdx<numPoints; ptIdx++) {
					logger.dbgLog(cName, mName, "Sparsity[" + prodIdx + ", " + ptIdx + "]=" + sparsityMatrix[prodIdx][ptIdx]);
//					logger.dbgLog(cName, mName, "Weight (" + ptIdx + ", " + ptIdx + ")=" + weightMatrix[ptIdx][ptIdx]);
				}
			}*/
			// Fill all points
			//
			//newPoints = fillPoints(points, sparsityMatrix, weightMatrix);
			//logger.dbgLog(cName, mName, "new points has " + newPoints.size() + " elements");
			// Update current points
			//
			points = newPoints;
			// Update distanceMatrix
			//
			distanceMatrix = points.getDistanceMatrix(propPower);

		} // End main loop
		if (propInstrumentation)
			instr.stopTiming();
		logger.dbgLog(cName, mName, "Took " + numIter + " iterations.");
		return newPoints;
	}
	
	/**
	 * Uses data filling to produce average product ratings for Recommendable products that don't have enough ratings.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A786F3101A5
	 */
	public final void fillInProdUCStats() throws AlkExcept 
	{
		final String mName = "fillInProdUCStats";
		Instrumenter inst1 = null, inst2 = null, inst3 = null;
		if (propInstrumentation) {
			inst1 = Instrumenter.getInstance(logger, cName, mName);
			inst2 = Instrumenter.getInstance(logger, cName, mName);
			inst3 = Instrumenter.getInstance(logger, cName, mName);
			inst1.startTiming();
		}
		try {
			for (int pcIdx = propStartPC; pcIdx <= propStopPC; pcIdx ++) {

				if (propInstrumentation) {
					inst2.startTiming();
					logger.log(cName, mName, "Starting PC " + pcIdx);
				}

				//	Retrieve the Core Product Means for this PC
				//
				VMeansList coreVMeans = cdb.getCoreVMeans(pcIdx);
				//logger.dbgLog(cName, mName, coreVMeans.dump());

				//	Number of points = number of UCs = number of VMeans in list
				//
				int numPoints = coreVMeans.size();

				//	Number of Core Products 
				//
				int numCoreProds = coreVMeans.get(0).numProducts();			

				//	Retrieve the Rec. Product Means for this PC
				//
				VMeansList recVMeans = cdb.getRecVMeans(pcIdx);
				//	Number of Rec. Products
				//			
				int numRecProds = recVMeans.get(0).numProducts();
				//	Array of Rec. Products
				//
				int[] recProds = recVMeans.get(0).getProductArray();

				//	Get number of members is each UC
				//
				int[] ucUsers = new int [numPoints];
				for (int pt = 0; pt < numPoints; pt ++) {
					ucUsers[pt] = cdb.getNumUsersInUC(pt, pcIdx);
				}
				if (propInstrumentation) 
					inst3.startTiming();

				//	Calculate the UC distance matrix using the Core Products
				//	Use symmetry (Dij = Dji) to reduce number of calcs.
				//
				double[][] distMatrix = coreVMeans.getDistanceMatrix(propPower);
				if (propInstrumentation)  {
					inst3.stopTiming("for distances");
					inst3.startTiming();
				}

				//	Calculate sparsity matrix for VMeans of 
				//	Recommendable Products
				//
				boolean[][] sparsityMatrix = recVMeans.getSparsityMatrix();
				//logger.dbgLog(cName, mName, "Finding points with missing data.");
				if (propInstrumentation)  {
					inst3.stopTiming("sparsity");
				}

				//	Set up new VMeansList for filled-in data
				//
				VMeansList newRecVMeans = null;
				newRecVMeans = (VMeansList)recVMeans.clone();
				//	Calculate Weights for points & fill in missing data
				//
				if (propInstrumentation)
					inst3.startTiming();
				double[] weights = new double[numPoints];
				for (int ptIdx1 = 0; ptIdx1 < numPoints; ptIdx1 ++ ) {
					Arrays.fill(weights, 0);
					for (int ptIdx2 = 0; ptIdx2 < numPoints; ptIdx2 ++) {
						//logger.dbgLog(cName, mName, "Entering inter-point loop: " + ipt2);
						// Weight of ipt2 proportional to cluster membership, inversely
						// proportion to distance between ipt2 and ipt.
						//	But if distance matrix element is zero, weight should also be zero.
						//
						if (ptIdx1 == ptIdx2) {
							weights[ptIdx2] = 0;
						}
						else if (distMatrix[ptIdx1][ptIdx2] == 0) {
							//logger.dbgLog(cName, mName, "Zero distance for points " + ptIdx1 + "," + ptIdx2);
							weights[ptIdx2]	= 0;
						}
						else {
							weights[ptIdx2] = (double)ucUsers[ptIdx2] / distMatrix[ptIdx1][ptIdx2];
						}
//						logger.dbgLog(cName, mName, "Weight[" + ipt + ", " + ipt2 + "]=" + weightMatrix[ipt][ipt2]);
					} // End loop over points from which we fill data (ipt)
					fillPoint(newRecVMeans.get(ptIdx1), recVMeans, sparsityMatrix, weights, ptIdx1);
				}
//				UserTransaction xa = getXA();
//				xa.begin();
				cdb.writeVMeansForPC(newRecVMeans, new ProductCluster(pcIdx));
//				xa.commit();				
				if (propInstrumentation) {
					inst3.stopTiming("Weights & filling");
					inst2.stopTiming("For PC " + pcIdx);
				}
			}
			
			if (propInstrumentation)
				inst1.stopTiming("For all PCs");
		}
		catch(Exception e) {
			AlkExcept ae = new AlkExcept(mName, 0, e);
			logger.err(cName, mName, ae.toString());
		}
	}
	
	/**
	 * Fills a single VMeans points from its neighbors and determines if the newly filled point is too far from the last iteration. The sparsity matrix and weight array together determine which neighbors to use and how much their evaluations should be weighted. This routine also performs tolerance checking on the newly filled point, since a copy of the old point is included in the neighbors list.
	 * @param pointToFill The point to fill.
	 * @param neighbors The list of points from which to fill data.
	 * @param sparsity The sparsity matrix calculated in fillFunction.
	 * @param pointWeights A part of the weight matrix, calculated in fillFunction, that pertains to the point to fill.
	 * @param ptNum The index of the point in its VMeansList.
	 * @return boolean
	 * @throws AlkExcept
	 * @roseuid 3A4B660903B9
	 */
	protected final boolean fillPoint(VMeans pointToFill, VMeansList neighbors, boolean[][] sparsity, double[] pointWeights, final int ptNum) throws AlkExcept 
	{
		final String mName = "fillPoints";
		int numProds =  neighbors.get(0).numProducts();
		int numneighbors = neighbors.size();
		boolean breaksTolerance = false;
		int[] prods = neighbors.get(0).getProductArray();		
		
		final int watchPoint = PropertyManager.getInt("debugWatchPoint");
		final int watchPID = PropertyManager.getInt("debugWatchPID");
		final boolean debugProdFill = PropertyManager.getBool("debugProdFill");
		// Loop over products
		//
		for (int prodIdx = 0; prodIdx < numProds; prodIdx ++) {
			int pid = prods[prodIdx];
			boolean watch = false;
			if (debugProdFill && pid == watchPID && ptNum == watchPoint) {
				watch = true;
				logger.log(cName, mName, "Called for point " + watchPoint + " for pid " + watchPID);
				StringBuffer buf = new StringBuffer("Weights:\n");
				for (int idx = 0; idx < pointWeights.length; idx++) {
					buf.append(idx);
					buf.append(": ");
					buf.append(pointWeights[idx]);
					buf.append("\n");						
				}
				logger.log(cName, mName, buf.toString());
			}
			//	Check weights to see if current point is missing 
			//	evaluation for product.
			//
			if (!sparsity[ptNum][prodIdx]) {
				
				if (watch) 
					logger.log(cName, mName, "Filling in pt " + ptNum + " for pid " + pid);

				//	Declare some storage for mean eval
				//	& set mean and weightSum to zero.
				//
				double mean = 0;
				double weightSum = 0;
				
				// Loop over other neighbors and fill in data.
				//
				for (int ipt2=0; ipt2<numneighbors; ipt2++) {
					// 	Skip "self" point
					//
					if (ipt2 == ptNum)
						continue;
					if (sparsity[ipt2][prodIdx]) {
						mean += neighbors.get(ipt2).getEval(pid) * pointWeights[ipt2];
						weightSum += pointWeights[ipt2];
					}
					if (watch) 
						logger.log(cName, mName, "Filling from pt " + ipt2 + " where eval = " + neighbors.get(ipt2).getEval(pid));
				} // End loop over other neighbors
				//logger.dbgLog(cName, mName, "prod=" + iprod + "pid=" + pd.id + " point=" + ipt + " weightSum=" + weightSum);
				// Normalize rating we just filled in (corr. to current point, current product).
				//
				if (watch) 
					logger.log(cName, mName, "meanSum=" + mean + " weightSum = " + weightSum);
				if (weightSum > 0.0) {
					mean /= weightSum;
					//fillneighbors.set(ipt, setEval(fillneighbors.get(ipt), iprod, (fillneighbors.get(ipt)).evaluation[iprod] / weightSum) );
				} // End if (weightSum > 0)
				
				//	Replace old mean with new one 
				//
				pointToFill.putEval(pid, (float)mean);
				if (watch) 
					logger.log(cName, mName, "Filled in pt " + ptNum + " with value " + mean);
				//	Test if new filled product mane breaks tolerance by
				//	comparing to same product mean in old points (neighbors).
				//
				if (!breaksTolerance && Math.abs(mean - neighbors.get(ptNum).getEval(pid)) > propDFTr)
					breaksTolerance = true;
			} // End if (current point missing evaluation for current product)
		} // End loop over products

		return breaksTolerance;
	}
	
	/**
	 * Constructs a list of UserCluster centers using evaluations of random users from the given UserRatingsList.
	 * @param url
	 * @return VMeansList
	 * @throws Exception
	 * @roseuid 3A4B66090271
	 */
	protected final VMeansList getInitialVMeans(UserRatingsList url) throws Exception 
	{
		final String mName = "getInitialVMeans";
		try {
			int vmeansidx = 0;
			UserRatings userRatings;

			// Random User Indexes from which Initial VMeans are cloned 
			//
			Random      rnduseridx         = new Random();    
			int         numusers          = url.numUsers();
	//       	int         numvmeans         = (int) numusers / AVG_NUM_USERS_IN_CLUSTER;
			int numvmeans = NUM_CLUSTERS;

			// #(Users Cloned) = #(VMeans Needed) 
			//
			int[] userIdxList = new int[numvmeans];    
			Arrays.fill(userIdxList, -1);
			while (vmeansidx < numvmeans) {
				int	rndidx = Math.abs(rnduseridx.nextInt(numusers));
				boolean addToList = true;
				//	Look for this user in list
				//
				for (int idx = 0; idx < numvmeans; idx++) {
					if (userIdxList[idx] == rndidx) {
						addToList = false;
						break;
					}
				}
						
				// If the User at given index has not already been cloned, add to the list. 
				//
				if (addToList) {
					userIdxList[vmeansidx] = rndidx;    
					vmeansidx++;
				}
			}
			
			VMeansList newPoints = new VMeansList(numvmeans);
			Object[] urAr = url.mapEntryArray();
			for (int j = 0; j < numvmeans; j++) {
				// 	Retrieve the users that have been randomly chosen to
				//	be replicated
				//
				Map.Entry me = (Map.Entry)urAr[userIdxList[j]];    
				userRatings = (UserRatings)me.getValue();
				//	For the selected person, put their evaluations for 
				//	all products into a VMeans object. 
				//
				VMeans aMean = new VMeans(url.numProducts());				
				//	Iterate over products & get this user's rating for each one.
				//	Also, stick the product IDs into an array to use in
				//	constructing the VMeans object. This is less than optimal,
				//	but will do until VMeans is changed.
				//
				//logger.dbgLog(cName, mName, "Iterating over " + url.numProducts() + " products");
				//	Iterator over Products and Ratings for the selected User
				//	Add the Product/Evaluation pair to the new VMeans
				//
				int numProds = userRatings.size();
				int[] prods = userRatings.getProductArray();
				float[] evals = userRatings.getEvalsArray();
				for (int idx = 0; idx < numProds; idx ++) {
					aMean.putEval(prods[idx], evals[idx]);
					//logger.dbgLog(cName, mName, "Adding pid=" + prod.id + " meaneval (D)=" + evalDouble);
				}
				//
				//logger.dbgLog(cName, mName, "Adding VMeans with " + aMean.numProducts() + " prods");
				//logger.dbgLog(cName, mName, aMean.dump());

				// Add cloned VMean to the Vector of VMeans
				//
				newPoints.add(aMean);    
			}
			//logger.dbgLog(cName, mName, newPoints.dump());
			return newPoints;
		}
		catch(NullPointerException ne) {
			ne.printStackTrace();
			throw new Exception(ne.getMessage());
		}
		catch (Exception e) {
			e.printStackTrace();
			throw e;
		}
	}
	
	/**
	 * @roseuid 3A8831DC0196
	 */
	protected VMeansList getRandomVMeans(UserRatingsList urList) throws AlkExcept 
	{
		VMeansList vml = new VMeansList(NUM_CLUSTERS);
		int numProds = urList.numProducts();
		for (int vmIdx = 0; vmIdx < NUM_CLUSTERS; vmIdx ++) {
			VMeans vm = new VMeans(numProds);
			Iterator prodIt = urList.products.iterator();
			while (prodIt.hasNext()) {
				Product pd = (Product)prodIt.next();
				vm.putEval(pd.id,  (float)Math.random() * (float)propMaxRating);
			}	//END loop over products
			vml.add(vm);
		}	//	END loop over VMeans
		return vml;
	}
	
	/**
	 * Intializes the set of UserClusters for all ProductClusters and stores their memberships in the database.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B66090252
	 */
	public void initializeClusters() throws AlkExcept 
	{
		
		//	Logging constants
		//
		final String mName = "initClusters";
		logger.log(cName, mName, "STARTING...");
		
		long mainStart = System.currentTimeMillis();
		Instrumenter instr = null;
		if (propInstrumentation) 
			instr = Instrumenter.getInstance(logger, cName, mName);
		java.text.NumberFormat numFormat = java.text.NumberFormat.getInstance();
		numFormat.setMinimumFractionDigits(4);
		numFormat.setMaximumFractionDigits(4);
		UserTransaction xa = null;
		try {
			for(int pcid = propStartPC; pcid <= propStopPC; pcid++) {
				long pcStart = System.currentTimeMillis();
				logger.dbgLog(cName, mName, "For Product Cluster..." + pcid);
				UserClusterList ucl = UserClusterList.getInstance();
				UserClusterList bestucl = null;
				VMeansList newVML = null; 
				VMeansList bestVML = null;
				VMeansList oldVML = null;
				double badrec;
				double bestbadrec = 5.0;  // assign a high value to insure replacement on first loop

				logger.dbgLog(cName, mName, "Total Number of Product Clusters: " + propNumPCs);
				
				logger.dbgLog(cName, mName, "getUserEvalsforPC...");
				if (propInstrumentation)
					instr.startTiming();
				ProductCluster pc = new ProductCluster(pcid);
				UserRatingsList url     = cdb.getCoreEvalsForPC(pc);   
				logger.dbgLog(cName, mName, "*** Main UserRatingsList has " + url.numUsers() + " users");
				if (propInstrumentation)
					instr.stopTiming();
				
				//	Outer Loop
				//
				for(int uberLoopIdx = 0; uberLoopIdx < propNumUberLoops; uberLoopIdx++) {
					logger.dbgLog(cName, mName, "Entering uberloop #" + uberLoopIdx);
					//logger.dbgLog(cName, mName, "now getting the initial cluster vmeans...");
					if (propUseRandomVMeans) {
						oldVML = getRandomVMeans(url);
					}
					else {
						try {
							oldVML = getInitialVMeans(url);
						}
						catch(Exception e) {
							throw new AlkExcept(e.toString(), 2102);
						}
					}
					//	Inner Loop - a single K-Means clustering run
					//
					boolean convergence = false;
					for (int innerLoopIdx = 0; innerLoopIdx < propNumInnerLoops && convergence==false; innerLoopIdx++){
						if (propInstrumentation)
							instr.startTiming();
						//	Fill the old VMeans if necessary (i.e., if not using totally random vmeans)
						//
						VMeansList oldVMLFilled = null;
						
						if (propUseRandomVMeans) {				
							oldVMLFilled = oldVML;
						}
						else {
							oldVMLFilled = fillFunction(oldVML, ucl);
						}
						//logger.dbgLog(cName, mName, "Original VMeans\n" + oldVML.dump());
						//	Cluster the users with these VMeans
						//
						ucl     = clusterUsers(oldVMLFilled, url); 
						//ucl = clusterUsers(oldVML, url);
						//	Recalc the VMeans using these cluster memberships
						//
						newVML = recalcVMeans(ucl);
						//logger.dbgLog(cName, mName, "Recalc'd VMeans:\n" + newVML.dump());
						
						//	Test for VMeans convergence and store the new VMeans as old VMeans for next iteration
						//
						convergence = testVMeansConvergence(oldVML, newVML);
						logger.dbgLog(cName, mName, "UberLoop #" + uberLoopIdx + " Mini-loop # " + innerLoopIdx + ": convergence is " + convergence);
						oldVML = newVML;
						//logger.dbgLog(cName, mName, oldVML.dump());
						if (propInstrumentation)
							instr.stopTiming("innerClusterLoop");
					}	//	END inner loop  

					//logger.dbgLog(cName, mName, "exiting while");

					// 	Evaluate how well we just clustered the users.
					//	Calculate % of bad recommendations.
					//
					badrec = 0;
					try {
						badrec = ucl.calcBadRecs(pcid);
					} catch (Exception e) {
						String errmsg = "Exception in calcBadRecs: " + e.getMessage();
						throw new AlkExcept(errmsg, 2101, e);
					}
					logger.dbgLog(cName, mName, "Badrecs=" + numFormat.format(badrec));
					//stopTiming(cName, mName);
					if (badrec < bestbadrec) { 
						logger.dbgLog(cName, mName, "New best badrec: previous best was " + numFormat.format(bestbadrec)); 
						bestbadrec = badrec;
						bestucl = ucl;
						bestVML = newVML;
					}
					
				} //END uberloop

				
				//	Output best badrec percentage:
				//
				logger.log(cName, mName, "FINAL Badrecs for PC #" + pcid + ": " + numFormat.format(bestbadrec));
				logger.log(cName, mName, "Calculating clusters took " + (System.currentTimeMillis() - pcStart) + " ms.");
				
				//	Make UserCluster List to save, using only UCs with members.
				//
				int uclSize = bestucl.size();
				UserClusterList uclToSave = UserClusterList.getInstance(uclSize);
				for (int ucIdx = 0; ucIdx < uclSize; ucIdx ++) {
					UserCluster uc = bestucl.get(ucIdx);
					if (uc.numUsers() > 0) {
						uclToSave.add(uc);
					}
				}
				bestucl = null;
				
				//	Make VMeansList to save from UCL to save & fill it in.
				//
				VMeansList vmlToSave = recalcVMeans(uclToSave);
				//logger.dbgLog(cName, mName, vmlToSave.dump());
				vmlToSave = fillFunction(vmlToSave, uclToSave);
				
				
				// Write best cluster information for the product cluster to the database.
				//
//				xa = getXA();
/*				try {
/					xa.begin();*/
					if (propInstrumentation)
						instr.startTiming();
					logger.dbgLog(cName, mName, "Writing UC List for PC #" + pcid);
					cdb.writeInitialUCForPC(uclToSave.size(),pc); 
					if (propInstrumentation) {
						instr.stopTiming();
						instr.startTiming();
					}
					logger.dbgLog(cName, mName, "Writing UC Membership for PC #" + pcid);
					cdb.writeUCMembershipForPC(uclToSave, pc);
					if (propInstrumentation) {
						instr.stopTiming();
						instr.startTiming();
					}
					logger.dbgLog(cName, mName, "Writing VMeans for PC #" + pcid);
					cdb.writeVMeansForPC(vmlToSave, pc);
					if (propInstrumentation) 
						instr.stopTiming();
/*					xa.commit();
				}
				catch(AlkExcept ae) {
					try { xa.rollback(); } catch (Exception e2) {}
					throw ae;
				}
				catch(Exception e) {
					try { xa.rollback(); } catch (Exception e2) {}
					throw e;
				}*/
			}
	//		logger.log(cName, mName, "Writing final recalcstats for all product clusters");
	//		recalcStats();        
			//logger.dbgLog(cName, mName, "goodbye...");
			long mainStop = System.currentTimeMillis();
			logger.log(cName, mName, "initializeClusters() took " + (mainStop - mainStart) + " ms.");
		}
		catch (AlkExcept ae) {
			logger.err(cName, mName, ae.toString());
			throw ae;
		}
		catch(Exception e) {
			AlkExcept ae2 = new AlkExcept(e.getMessage(), 2103, e);
			logger.err(cName, mName, ae2.toString());
			throw ae2;
		}
	}
	
	/**
	 * @return void
	 * @roseuid 3A4B66090242
	 */
	public final void initLogging() 
	{
		if (logger == null) {
			logger = new LogManager();
		}
	}
	
	/**
	 * Reads properties from the current Context environment.
	 * @return void
	 * @roseuid 3A4B66090251
	 */
	protected final void readProps() 
	{
		final String mName = "readProps";
		try {
			propNumPCs = PropertyManager.getInt("NumPC");
			propNumUberLoops = PropertyManager.getInt("UberLoops");
			propNumInnerLoops = PropertyManager.getInt("I");
			propPower = PropertyManager.getFloat("P");
			propOmega = PropertyManager.getInt("Omega");
			propTopNProds = PropertyManager.getInt("TopNProds");
			propMaxDFTIter = PropertyManager.getInt("MAX_DFT_iter");
			propDFTr = PropertyManager.getDouble("DFT_r");
			propTr = PropertyManager.getDouble("Tr");
			propStartPC = PropertyManager.getInt("startPC");
			propStopPC = PropertyManager.getInt("stopPC");
			propInstrumentation = PropertyManager.getBool("Instrumentation");
			propUseRandomVMeans = PropertyManager.getBool("useRandomVMeans");
			propMaxRating = PropertyManager.getDouble("MaxRating");
			cdb = new ClusterMgrDBUtils(logger);
		}
		catch(Exception e) {
			//logger.err(cName, mName, e.getMessage());
			System.out.println("ERROR in " + cName + "." + mName + ": " + e.getMessage());
			//e.printStackTrace();
		}
	}
	
	/**
	 * @roseuid 3A4B660903A9
	 */
	public void recalcProductStats() throws AlkExcept 
	{
	/*	cdb.recalcProdStat();
		cdb.recalcUCStat();
		cdb.recalcProdUCStat(0);
		cdb.recalcProdScore(0);
		cdb.recalcPCStat(propOmega, propTopNProds);
		cdb.recalcProdPCStat();*/
	}
	
	/**
	 * Recalculates all statistics which depend in UserCluster membership.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B66090399
	 */
	public void recalcStats() throws AlkExcept 
	{
		final String mName = "recalcStats";
		

		boolean debugProdFill = PropertyManager.getBool("debugProdFill");
		if (debugProdFill) { 
			logger.dbgLog(cName, mName, "Stopping after fillInProdUCStats");
			fillInProdUCStats();
			logger.log(cName, mName, "Finished.");
			return;
		}
		try {
			logger.log(cName, mName, "Starting.");
			cdb.updateUserSubGroups();
			cdb.recalcProdStat();
			cdb.recalcUserStat();
			cdb.recalcUCStat();
			cdb.recalcProdUCStat(new ProductCluster(0));
			fillInProdUCStats();
			if (debugProdFill) {
				logger.log(cName, mName, "Finished.");
				return;
			}
//			cdb.recalcProdScore(new ProductCluster(0));
			cdb.recalcPCStat(propOmega, propTopNProds);
			cdb.recalcUserPCStat(propOmega);
			cdb.recalcUserUCStat();
			cdb.recalcUserSFStat();
			cdb.recalcProdPCStat();
			cdb.recalcUserDataStat();
			cdb.recalcAlkindexStat();
			logger.log(cName, mName, "Finished.");
		}
		catch (AlkExcept ae) {
			logger.err(cName, mName, ae.toString());
		}
		catch (Exception e) {
			logger.err(cName, mName, e.getMessage());
		}
	}
	
	/**
	 * Recalculates the cluser centers of the given UserClusterList.
	 * @param clusterList
	 * @return VMeansList
	 * @throws AlkExcept
	 * @roseuid 3A4B6609032C
	 */
	protected final VMeansList recalcVMeans(final UserClusterList clusterList) throws AlkExcept 
	{
		final String mName = "recalcVMeans";
		Instrumenter instr = null;
		if (propInstrumentation) {
			instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
		}
		UserCluster aCluster = null;
        VMeansList  newPoints = new VMeansList(clusterList.size());

		//	Iterate over user clusters
		//
		int numClus = clusterList.size();
		for (int clusIdx = 0; clusIdx < numClus; clusIdx++) {
			aCluster = clusterList.get(clusIdx);
			aCluster.recalcVMeans();
			newPoints.add(aCluster.vMeans);

       	} 	//END iterate over UserClusters
       	if (propInstrumentation)
	       	instr.stopTiming();

		return newPoints;
	}
	
	/**
	 * Rebuilds the cluster and stores new memberships in the database. Similar to intiializeClusters, except that it uses the current cluster memberships as a starting point.
	 * @roseuid 3A4B66090261
	 */
	public void regenerateClusters() throws AlkExcept 
	{

		//	Logging constants
		//
		final String cName = "CM";
		final String mName = "regenClusters";
		boolean convergence;
		cdb.delRelUserCluster();

		logger.log(cName, mName, "STARTING...");

		Instrumenter inst = null;

		try {
			if (propInstrumentation) {
				inst = Instrumenter.getInstance(logger, cName, mName);
				inst.startTiming();
			}
			for(int pcid = propStartPC; pcid <= propStopPC; pcid++) {
				UserClusterList ucl = null;
				VMeansList oldVML = null;
				VMeansList newVML = null;
				ProductCluster pc = new ProductCluster(pcid);
				//	Read existing VMeans from the PC
				//
				logger.dbgLog(cName, mName, "For Product Cluster..." + pcid);
				logger.dbgLog(cName, mName, "Reading existing VMeans For PC " + pcid);
				oldVML = cdb.getCoreVMeans(pcid);    

				//	Fill the Old VMeans -- NOTE: VMeans are already filled!
				//
				//logger.dbgLog(cName, mName, "filling oldvml...");
				//oldVML = fillFunction(oldVML, ucl);	
				//newVML = recalcVMeans(oldUCL, oldVML);
				//oldVML = newVML;

				//	Get evaluations for the PC
				//
				UserRatingsList url     = cdb.getCoreEvalsForPC(pc);    

				int j = 0;

				do {
					oldVML = fillFunction(oldVML, ucl);	
					logger.dbgLog(cName, mName, "Clustering all users...");
					ucl     = clusterUsers(oldVML, url); 
					logger.dbgLog(cName, mName, "recalcVMeans...");
					newVML = recalcVMeans(ucl);
					convergence = testVMeansConvergence(oldVML, newVML);
					logger.dbgLog(cName, mName, "Convergence is " + convergence);
					oldVML = newVML;
					j++;

				} while ((convergence == false) && (j < propNumInnerLoops)); 
				
				//	Write filled VMeans
				//
				VMeansList newVMLFilled = fillFunction(newVML, ucl);
				//UserTransaction xa = getXA();
//				try {
				//	xa.begin();
					logger.dbgLog(cName, mName, "Writing UC Membership for PC #" + pcid);
					cdb.writeUCMembershipForPC(ucl, pc);
					logger.dbgLog(cName, mName, "Writing VMeans for PC #" + pcid);
					cdb.writeVMeansForPC(newVMLFilled, pc);
				//	xa.commit();
/*				}
				catch(AlkExcept ae) {
					try { xa.rollback(); } catch(Exception e2) {}
					throw ae;
				}
				catch(Exception e) {
					try { xa.rollback(); } catch(Exception e2) {}
					throw e;
				}*/
				
			}
			if (propInstrumentation)
				inst.stopTiming();

			logger.log(cName, mName, "Finished.");
			//logger.dbgLog(cName, mName, "recalcStats...");
			//recalcStats();
		}
		catch(AlkExcept ae) {
			logger.err(cName, mName, ae.toString());
			throw ae;
		}
		catch(Exception e) {
			AlkExcept ae2 = new AlkExcept("", 2122, e);
			logger.err(cName, mName, ae2.toString());
			throw ae2;			
		}
	}
	
	/**
	 * Determines whether two VMeans are too far apart.
	 * @param oldvml
	 * @param newvml
	 * @return boolean
	 * @throws AlkExcept
	 * @roseuid 3A4B6609035B
	 */
	protected final boolean testVMeansConvergence(VMeansList oldvml, VMeansList newvml) throws AlkExcept 
	{
		
		final String mName = "testVMeansConvergence";
		Instrumenter instr = null;
		if (propInstrumentation) {
			instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
		}
		// Array for Max. of differences between movies belonging to old and new VMeans
		//
		int numVMeans = oldvml.size();
   		double[] max_delta_mean = new double[numVMeans];    

		// Iterate through all updated Vmeans
		//
		for (int vmIdx = 0; vmIdx < numVMeans; vmIdx++) {    
			VMeans oldmean = (VMeans)oldvml.get(vmIdx);
			VMeans newmean = (VMeans)newvml.get(vmIdx);
			
			// 	Compare the mean eval for each Product at this point.
			//	Convergence is true when the difference is smaller than
			//	the environment property Tr for each Product at each point.
			//
			int[] prods = oldmean.getProductArray();
			float[] oldEvals = oldmean.getEvalsArray();
			int numProds = oldmean.size();
			for (int idx = 0; idx < numProds; idx ++ ){
				// Compute difference between new and old VMeans in 1-dim. 
				//
				double vmeanDiff = Math.abs(newmean.getEval(prods[idx]) - oldEvals[idx]);    
				if (vmeanDiff > propTr) {
				//	logger.dbgLog(cName, mName,"delta =  " + vmeanDiff + " POINT=" + vmIdx + " PRODID=" + pd.id + " old eval=" + oldmean.getEvalValue(pd) + " new eval=" + newmean.getEvalValue(pd));
					if (propInstrumentation)
						instr.stopTiming();
					return false;
				}
			}
		 }
		 if (propInstrumentation)
			 instr.stopTiming();

		return true;
	}
	
	/**
	 * Retrieves a UserTransaction. This method must be implemented by any class extending this class.
	 * @return A UserTransaction for use by the object.
	 * @roseuid 3AAFDC030177
	 */
	public abstract javax.transaction.UserTransaction getXA();
}
